"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const console_1 = require("console");
const util_1 = __importDefault(require("util"));
const session = __importStar(require("express-session"));
const mongodb_1 = require("mongodb");
const debug_1 = __importDefault(require("debug"));
const debug = debug_1.default('connect-mongo');
// eslint-disable-next-line @typescript-eslint/no-empty-function
const noop = () => { };
const unit = (a) => a;
function defaultSerializeFunction(session) {
    // Copy each property of the session to a new object
    const obj = {};
    let prop;
    for (prop in session) {
        if (prop === 'cookie') {
            // Convert the cookie instance to an object, if possible
            // This gets rid of the duplicate object under session.cookie.data property
            // @ts-ignore FIXME:
            obj.cookie = session.cookie.toJSON
                ? // @ts-ignore FIXME:
                    session.cookie.toJSON()
                : session.cookie;
        }
        else {
            // @ts-ignore FIXME:
            obj[prop] = session[prop];
        }
    }
    return obj;
}
function computeTransformFunctions(options) {
    if (options.serialize || options.unserialize) {
        return {
            serialize: options.serialize || defaultSerializeFunction,
            unserialize: options.unserialize || unit,
        };
    }
    if (options.stringify === false) {
        return {
            serialize: defaultSerializeFunction,
            unserialize: unit,
        };
    }
    // Default case
    return {
        serialize: JSON.stringify,
        unserialize: JSON.parse,
    };
}
class MongoStore extends session.Store {
    constructor({ collectionName = 'sessions', ttl = 1209600, mongoOptions = {}, autoRemove = 'native', autoRemoveInterval = 10, touchAfter = 0, stringify = true, crypto, ...required }) {
        super();
        this.crypto = null;
        debug('create MongoStore instance');
        const options = {
            collectionName,
            ttl,
            mongoOptions,
            autoRemove,
            autoRemoveInterval,
            touchAfter,
            stringify,
            crypto: {
                ...{
                    secret: false,
                    algorithm: 'aes-256-gcm',
                    hashing: 'sha512',
                    encodeas: 'base64',
                    key_size: 32,
                    iv_size: 16,
                    at_size: 16,
                },
                ...crypto,
            },
            ...required,
        };
        // Check params
        console_1.assert(options.mongoUrl || options.clientPromise || options.client, 'You must provide either mongoUrl|clientPromise|client in options');
        console_1.assert(options.createAutoRemoveIdx === null ||
            options.createAutoRemoveIdx === undefined, 'options.createAutoRemoveIdx has been reverted to autoRemove and autoRemoveInterval');
        console_1.assert(!options.autoRemoveInterval || options.autoRemoveInterval <= 71582, 
        /* (Math.pow(2, 32) - 1) / (1000 * 60) */ 'autoRemoveInterval is too large. options.autoRemoveInterval is in minutes but not seconds nor mills');
        this.transformFunctions = computeTransformFunctions(options);
        let _clientP;
        if (options.mongoUrl) {
            _clientP = mongodb_1.MongoClient.connect(options.mongoUrl, options.mongoOptions);
        }
        else if (options.clientPromise) {
            _clientP = options.clientPromise;
        }
        else if (options.client) {
            _clientP = Promise.resolve(options.client);
        }
        else {
            throw new Error('Cannot init client. Please provide correct options');
        }
        console_1.assert(!!_clientP, 'Client is null|undefined');
        this.clientP = _clientP;
        this.options = options;
        this.collectionP = _clientP.then(async (con) => {
            const collection = con
                .db(options.dbName)
                .collection(options.collectionName);
            await this.setAutoRemove(collection);
            return collection;
        });
        if (options.crypto.secret) {
            this.crypto = require('kruptein')(options.crypto);
        }
    }
    static create(options) {
        return new MongoStore(options);
    }
    setAutoRemove(collection) {
        const removeQuery = () => ({
            expires: {
                $lt: new Date(),
            },
        });
        switch (this.options.autoRemove) {
            case 'native':
                debug('Creating MongoDB TTL index');
                return collection.createIndex({ expires: 1 }, {
                    background: true,
                    expireAfterSeconds: 0,
                });
            case 'interval':
                debug('create Timer to remove expired sessions');
                this.timer = setInterval(() => collection.deleteMany(removeQuery(), {
                    writeConcern: {
                        w: 0,
                        j: false,
                    },
                }), this.options.autoRemoveInterval * 1000 * 60);
                this.timer.unref();
                return Promise.resolve();
            case 'disabled':
            default:
                return Promise.resolve();
        }
    }
    computeStorageId(sessionId) {
        if (this.options.transformId &&
            typeof this.options.transformId === 'function') {
            return this.options.transformId(sessionId);
        }
        return sessionId;
    }
    /**
     * promisify and bind the `this.crypto.get` function.
     * Please check !!this.crypto === true before using this getter!
     */
    get cryptoGet() {
        if (!this.crypto) {
            throw new Error('Check this.crypto before calling this.cryptoGet!');
        }
        return util_1.default.promisify(this.crypto.get).bind(this.crypto);
    }
    /**
     * Decrypt given session data
     * @param session session data to be decrypt. Mutate the input session.
     */
    async decryptSession(session) {
        if (this.crypto && session) {
            const plaintext = await this.cryptoGet(this.options.crypto.secret, session.session).catch((err) => {
                throw new Error(err);
            });
            // @ts-ignore
            session.session = JSON.parse(plaintext);
        }
    }
    /**
     * Get a session from the store given a session ID (sid)
     * @param sid session ID
     */
    get(sid, callback) {
        ;
        (async () => {
            try {
                debug(`MongoStore#get=${sid}`);
                const collection = await this.collectionP;
                const session = await collection.findOne({
                    _id: this.computeStorageId(sid),
                    $or: [
                        { expires: { $exists: false } },
                        { expires: { $gt: new Date() } },
                    ],
                });
                if (this.crypto && session) {
                    await this.decryptSession(session).catch((err) => callback(err));
                }
                const s = session && this.transformFunctions.unserialize(session.session);
                if (this.options.touchAfter > 0 && (session === null || session === void 0 ? void 0 : session.lastModified)) {
                    s.lastModified = session.lastModified;
                }
                this.emit('get', sid);
                callback(null, s === undefined ? null : s);
            }
            catch (error) {
                callback(error);
            }
        })();
    }
    /**
     * Upsert a session into the store given a session ID (sid) and session (session) object.
     * @param sid session ID
     * @param session session object
     */
    set(sid, session, callback = noop) {
        ;
        (async () => {
            var _a;
            try {
                debug(`MongoStore#set=${sid}`);
                // Removing the lastModified prop from the session object before update
                // @ts-ignore
                if (this.options.touchAfter > 0 && (session === null || session === void 0 ? void 0 : session.lastModified)) {
                    // @ts-ignore
                    delete session.lastModified;
                }
                const s = {
                    _id: this.computeStorageId(sid),
                    session: this.transformFunctions.serialize(session),
                };
                // Expire handling
                if ((_a = session === null || session === void 0 ? void 0 : session.cookie) === null || _a === void 0 ? void 0 : _a.expires) {
                    s.expires = new Date(session.cookie.expires);
                }
                else {
                    // If there's no expiration date specified, it is
                    // browser-session cookie or there is no cookie at all,
                    // as per the connect docs.
                    //
                    // So we set the expiration to two-weeks from now
                    // - as is common practice in the industry (e.g Django) -
                    // or the default specified in the options.
                    s.expires = new Date(Date.now() + this.options.ttl * 1000);
                }
                // Last modify handling
                if (this.options.touchAfter > 0) {
                    s.lastModified = new Date();
                }
                if (this.crypto) {
                    const cryptoSet = util_1.default.promisify(this.crypto.set).bind(this.crypto);
                    const data = await cryptoSet(this.options.crypto.secret, s.session).catch((err) => {
                        throw new Error(err);
                    });
                    s.session = data;
                }
                const collection = await this.collectionP;
                const rawResp = await collection.updateOne({ _id: s._id }, { $set: s }, {
                    upsert: true,
                    writeConcern: this.options.writeOperationOptions,
                });
                if (rawResp.upsertedCount > 0) {
                    this.emit('create', sid);
                }
                else {
                    this.emit('update', sid);
                }
                this.emit('set', sid);
            }
            catch (error) {
                return callback(error);
            }
            return callback(null);
        })();
    }
    touch(sid, session, callback = noop) {
        ;
        (async () => {
            var _a;
            try {
                debug(`MongoStore#touch=${sid}`);
                const updateFields = {};
                const touchAfter = this.options.touchAfter * 1000;
                const lastModified = session.lastModified
                    ? session.lastModified.getTime()
                    : 0;
                const currentDate = new Date();
                // If the given options has a touchAfter property, check if the
                // current timestamp - lastModified timestamp is bigger than
                // the specified, if it's not, don't touch the session
                if (touchAfter > 0 && lastModified > 0) {
                    const timeElapsed = currentDate.getTime() - lastModified;
                    if (timeElapsed < touchAfter) {
                        debug(`Skip touching session=${sid}`);
                        return callback(null);
                    }
                    updateFields.lastModified = currentDate;
                }
                if ((_a = session === null || session === void 0 ? void 0 : session.cookie) === null || _a === void 0 ? void 0 : _a.expires) {
                    updateFields.expires = new Date(session.cookie.expires);
                }
                else {
                    updateFields.expires = new Date(Date.now() + this.options.ttl * 1000);
                }
                const collection = await this.collectionP;
                const rawResp = await collection.updateOne({ _id: this.computeStorageId(sid) }, { $set: updateFields }, { writeConcern: this.options.writeOperationOptions });
                if (rawResp.matchedCount === 0) {
                    return callback(new Error('Unable to find the session to touch'));
                }
                else {
                    this.emit('touch', sid, session);
                    return callback(null);
                }
            }
            catch (error) {
                return callback(error);
            }
        })();
    }
    /**
     * Get all sessions in the store as an array
     */
    all(callback) {
        ;
        (async () => {
            try {
                debug('MongoStore#all()');
                const collection = await this.collectionP;
                const sessions = collection.find({
                    $or: [
                        { expires: { $exists: false } },
                        { expires: { $gt: new Date() } },
                    ],
                });
                const results = [];
                for await (const session of sessions) {
                    if (this.crypto && session) {
                        await this.decryptSession(session);
                    }
                    results.push(this.transformFunctions.unserialize(session.session));
                }
                this.emit('all', results);
                callback(null, results);
            }
            catch (error) {
                callback(error);
            }
        })();
    }
    /**
     * Destroy/delete a session from the store given a session ID (sid)
     * @param sid session ID
     */
    destroy(sid, callback = noop) {
        debug(`MongoStore#destroy=${sid}`);
        this.collectionP
            .then((colleciton) => colleciton.deleteOne({ _id: this.computeStorageId(sid) }, { writeConcern: this.options.writeOperationOptions }))
            .then(() => {
            this.emit('destroy', sid);
            callback(null);
        })
            .catch((err) => callback(err));
    }
    /**
     * Get the count of all sessions in the store
     */
    length(callback) {
        debug('MongoStore#length()');
        this.collectionP
            .then((collection) => collection.countDocuments())
            .then((c) => callback(null, c))
            // @ts-ignore
            .catch((err) => callback(err));
    }
    /**
     * Delete all sessions from the store.
     */
    clear(callback = noop) {
        debug('MongoStore#clear()');
        this.collectionP
            .then((collection) => collection.drop())
            .then(() => callback(null))
            .catch((err) => callback(err));
    }
    /**
     * Close database connection
     */
    close() {
        debug('MongoStore#close()');
        return this.clientP.then((c) => c.close());
    }
}
exports.default = MongoStore;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiTW9uZ29TdG9yZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9saWIvTW9uZ29TdG9yZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7QUFBQSxxQ0FBZ0M7QUFDaEMsZ0RBQXVCO0FBQ3ZCLHlEQUEwQztBQUMxQyxxQ0FLZ0I7QUFDaEIsa0RBQXlCO0FBR3pCLE1BQU0sS0FBSyxHQUFHLGVBQUssQ0FBQyxlQUFlLENBQUMsQ0FBQTtBQWdFcEMsZ0VBQWdFO0FBQ2hFLE1BQU0sSUFBSSxHQUFHLEdBQUcsRUFBRSxHQUFFLENBQUMsQ0FBQTtBQUNyQixNQUFNLElBQUksR0FBbUIsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQTtBQUVyQyxTQUFTLHdCQUF3QixDQUMvQixPQUE0QjtJQUU1QixvREFBb0Q7SUFDcEQsTUFBTSxHQUFHLEdBQUcsRUFBRSxDQUFBO0lBQ2QsSUFBSSxJQUFJLENBQUE7SUFDUixLQUFLLElBQUksSUFBSSxPQUFPLEVBQUU7UUFDcEIsSUFBSSxJQUFJLEtBQUssUUFBUSxFQUFFO1lBQ3JCLHdEQUF3RDtZQUN4RCwyRUFBMkU7WUFDM0Usb0JBQW9CO1lBQ3BCLEdBQUcsQ0FBQyxNQUFNLEdBQUcsT0FBTyxDQUFDLE1BQU0sQ0FBQyxNQUFNO2dCQUNoQyxDQUFDLENBQUMsb0JBQW9CO29CQUNwQixPQUFPLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRTtnQkFDekIsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUE7U0FDbkI7YUFBTTtZQUNMLG9CQUFvQjtZQUNwQixHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFBO1NBQzFCO0tBQ0Y7SUFFRCxPQUFPLEdBQTBCLENBQUE7QUFDbkMsQ0FBQztBQUVELFNBQVMseUJBQXlCLENBQUMsT0FBbUM7SUFDcEUsSUFBSSxPQUFPLENBQUMsU0FBUyxJQUFJLE9BQU8sQ0FBQyxXQUFXLEVBQUU7UUFDNUMsT0FBTztZQUNMLFNBQVMsRUFBRSxPQUFPLENBQUMsU0FBUyxJQUFJLHdCQUF3QjtZQUN4RCxXQUFXLEVBQUUsT0FBTyxDQUFDLFdBQVcsSUFBSSxJQUFJO1NBQ3pDLENBQUE7S0FDRjtJQUVELElBQUksT0FBTyxDQUFDLFNBQVMsS0FBSyxLQUFLLEVBQUU7UUFDL0IsT0FBTztZQUNMLFNBQVMsRUFBRSx3QkFBd0I7WUFDbkMsV0FBVyxFQUFFLElBQUk7U0FDbEIsQ0FBQTtLQUNGO0lBQ0QsZUFBZTtJQUNmLE9BQU87UUFDTCxTQUFTLEVBQUUsSUFBSSxDQUFDLFNBQVM7UUFDekIsV0FBVyxFQUFFLElBQUksQ0FBQyxLQUFLO0tBQ3hCLENBQUE7QUFDSCxDQUFDO0FBRUQsTUFBcUIsVUFBVyxTQUFRLE9BQU8sQ0FBQyxLQUFLO0lBWW5ELFlBQVksRUFDVixjQUFjLEdBQUcsVUFBVSxFQUMzQixHQUFHLEdBQUcsT0FBTyxFQUNiLFlBQVksR0FBRyxFQUFFLEVBQ2pCLFVBQVUsR0FBRyxRQUFRLEVBQ3JCLGtCQUFrQixHQUFHLEVBQUUsRUFDdkIsVUFBVSxHQUFHLENBQUMsRUFDZCxTQUFTLEdBQUcsSUFBSSxFQUNoQixNQUFNLEVBQ04sR0FBRyxRQUFRLEVBQ1M7UUFDcEIsS0FBSyxFQUFFLENBQUE7UUFyQkQsV0FBTSxHQUFvQixJQUFJLENBQUE7UUFzQnBDLEtBQUssQ0FBQyw0QkFBNEIsQ0FBQyxDQUFBO1FBQ25DLE1BQU0sT0FBTyxHQUErQjtZQUMxQyxjQUFjO1lBQ2QsR0FBRztZQUNILFlBQVk7WUFDWixVQUFVO1lBQ1Ysa0JBQWtCO1lBQ2xCLFVBQVU7WUFDVixTQUFTO1lBQ1QsTUFBTSxFQUFFO2dCQUNOLEdBQUc7b0JBQ0QsTUFBTSxFQUFFLEtBQUs7b0JBQ2IsU0FBUyxFQUFFLGFBQWE7b0JBQ3hCLE9BQU8sRUFBRSxRQUFRO29CQUNqQixRQUFRLEVBQUUsUUFBUTtvQkFDbEIsUUFBUSxFQUFFLEVBQUU7b0JBQ1osT0FBTyxFQUFFLEVBQUU7b0JBQ1gsT0FBTyxFQUFFLEVBQUU7aUJBQ1o7Z0JBQ0QsR0FBRyxNQUFNO2FBQ1Y7WUFDRCxHQUFHLFFBQVE7U0FDWixDQUFBO1FBQ0QsZUFBZTtRQUNmLGdCQUFNLENBQ0osT0FBTyxDQUFDLFFBQVEsSUFBSSxPQUFPLENBQUMsYUFBYSxJQUFJLE9BQU8sQ0FBQyxNQUFNLEVBQzNELGtFQUFrRSxDQUNuRSxDQUFBO1FBQ0QsZ0JBQU0sQ0FDSixPQUFPLENBQUMsbUJBQW1CLEtBQUssSUFBSTtZQUNsQyxPQUFPLENBQUMsbUJBQW1CLEtBQUssU0FBUyxFQUMzQyxvRkFBb0YsQ0FDckYsQ0FBQTtRQUNELGdCQUFNLENBQ0osQ0FBQyxPQUFPLENBQUMsa0JBQWtCLElBQUksT0FBTyxDQUFDLGtCQUFrQixJQUFJLEtBQUs7UUFDbEUseUNBQXlDLENBQUMscUdBQXFHLENBQ2hKLENBQUE7UUFDRCxJQUFJLENBQUMsa0JBQWtCLEdBQUcseUJBQXlCLENBQUMsT0FBTyxDQUFDLENBQUE7UUFDNUQsSUFBSSxRQUE4QixDQUFBO1FBQ2xDLElBQUksT0FBTyxDQUFDLFFBQVEsRUFBRTtZQUNwQixRQUFRLEdBQUcscUJBQVcsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLFFBQVEsRUFBRSxPQUFPLENBQUMsWUFBWSxDQUFDLENBQUE7U0FDdkU7YUFBTSxJQUFJLE9BQU8sQ0FBQyxhQUFhLEVBQUU7WUFDaEMsUUFBUSxHQUFHLE9BQU8sQ0FBQyxhQUFhLENBQUE7U0FDakM7YUFBTSxJQUFJLE9BQU8sQ0FBQyxNQUFNLEVBQUU7WUFDekIsUUFBUSxHQUFHLE9BQU8sQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFBO1NBQzNDO2FBQU07WUFDTCxNQUFNLElBQUksS0FBSyxDQUFDLG9EQUFvRCxDQUFDLENBQUE7U0FDdEU7UUFDRCxnQkFBTSxDQUFDLENBQUMsQ0FBQyxRQUFRLEVBQUUsMEJBQTBCLENBQUMsQ0FBQTtRQUM5QyxJQUFJLENBQUMsT0FBTyxHQUFHLFFBQVEsQ0FBQTtRQUN2QixJQUFJLENBQUMsT0FBTyxHQUFHLE9BQU8sQ0FBQTtRQUN0QixJQUFJLENBQUMsV0FBVyxHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLEdBQUcsRUFBRSxFQUFFO1lBQzdDLE1BQU0sVUFBVSxHQUFHLEdBQUc7aUJBQ25CLEVBQUUsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDO2lCQUNsQixVQUFVLENBQXNCLE9BQU8sQ0FBQyxjQUFjLENBQUMsQ0FBQTtZQUMxRCxNQUFNLElBQUksQ0FBQyxhQUFhLENBQUMsVUFBVSxDQUFDLENBQUE7WUFDcEMsT0FBTyxVQUFVLENBQUE7UUFDbkIsQ0FBQyxDQUFDLENBQUE7UUFDRixJQUFJLE9BQU8sQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFO1lBQ3pCLElBQUksQ0FBQyxNQUFNLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsQ0FBQTtTQUNsRDtJQUNILENBQUM7SUFFRCxNQUFNLENBQUMsTUFBTSxDQUFDLE9BQTRCO1FBQ3hDLE9BQU8sSUFBSSxVQUFVLENBQUMsT0FBTyxDQUFDLENBQUE7SUFDaEMsQ0FBQztJQUVPLGFBQWEsQ0FDbkIsVUFBMkM7UUFFM0MsTUFBTSxXQUFXLEdBQUcsR0FBRyxFQUFFLENBQUMsQ0FBQztZQUN6QixPQUFPLEVBQUU7Z0JBQ1AsR0FBRyxFQUFFLElBQUksSUFBSSxFQUFFO2FBQ2hCO1NBQ0YsQ0FBQyxDQUFBO1FBQ0YsUUFBUSxJQUFJLENBQUMsT0FBTyxDQUFDLFVBQVUsRUFBRTtZQUMvQixLQUFLLFFBQVE7Z0JBQ1gsS0FBSyxDQUFDLDRCQUE0QixDQUFDLENBQUE7Z0JBQ25DLE9BQU8sVUFBVSxDQUFDLFdBQVcsQ0FDM0IsRUFBRSxPQUFPLEVBQUUsQ0FBQyxFQUFFLEVBQ2Q7b0JBQ0UsVUFBVSxFQUFFLElBQUk7b0JBQ2hCLGtCQUFrQixFQUFFLENBQUM7aUJBQ3RCLENBQ0YsQ0FBQTtZQUNILEtBQUssVUFBVTtnQkFDYixLQUFLLENBQUMseUNBQXlDLENBQUMsQ0FBQTtnQkFDaEQsSUFBSSxDQUFDLEtBQUssR0FBRyxXQUFXLENBQ3RCLEdBQUcsRUFBRSxDQUNILFVBQVUsQ0FBQyxVQUFVLENBQUMsV0FBVyxFQUFFLEVBQUU7b0JBQ25DLFlBQVksRUFBRTt3QkFDWixDQUFDLEVBQUUsQ0FBQzt3QkFDSixDQUFDLEVBQUUsS0FBSztxQkFDVDtpQkFDRixDQUFDLEVBQ0osSUFBSSxDQUFDLE9BQU8sQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLEdBQUcsRUFBRSxDQUM1QyxDQUFBO2dCQUNELElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLENBQUE7Z0JBQ2xCLE9BQU8sT0FBTyxDQUFDLE9BQU8sRUFBRSxDQUFBO1lBQzFCLEtBQUssVUFBVSxDQUFDO1lBQ2hCO2dCQUNFLE9BQU8sT0FBTyxDQUFDLE9BQU8sRUFBRSxDQUFBO1NBQzNCO0lBQ0gsQ0FBQztJQUVPLGdCQUFnQixDQUFDLFNBQWlCO1FBQ3hDLElBQ0UsSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1lBQ3hCLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXLEtBQUssVUFBVSxFQUM5QztZQUNBLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLENBQUE7U0FDM0M7UUFDRCxPQUFPLFNBQVMsQ0FBQTtJQUNsQixDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsSUFBWSxTQUFTO1FBQ25CLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFO1lBQ2hCLE1BQU0sSUFBSSxLQUFLLENBQUMsa0RBQWtELENBQUMsQ0FBQTtTQUNwRTtRQUNELE9BQU8sY0FBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUE7SUFDMUQsQ0FBQztJQUVEOzs7T0FHRztJQUNLLEtBQUssQ0FBQyxjQUFjLENBQzFCLE9BQStDO1FBRS9DLElBQUksSUFBSSxDQUFDLE1BQU0sSUFBSSxPQUFPLEVBQUU7WUFDMUIsTUFBTSxTQUFTLEdBQUcsTUFBTSxJQUFJLENBQUMsU0FBUyxDQUNwQyxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxNQUFnQixFQUNwQyxPQUFPLENBQUMsT0FBTyxDQUNoQixDQUFDLEtBQUssQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFO2dCQUNkLE1BQU0sSUFBSSxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUE7WUFDdEIsQ0FBQyxDQUFDLENBQUE7WUFDRixhQUFhO1lBQ2IsT0FBTyxDQUFDLE9BQU8sR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFBO1NBQ3hDO0lBQ0gsQ0FBQztJQUVEOzs7T0FHRztJQUNILEdBQUcsQ0FDRCxHQUFXLEVBQ1gsUUFBa0U7UUFFbEUsQ0FBQztRQUFBLENBQUMsS0FBSyxJQUFJLEVBQUU7WUFDWCxJQUFJO2dCQUNGLEtBQUssQ0FBQyxrQkFBa0IsR0FBRyxFQUFFLENBQUMsQ0FBQTtnQkFDOUIsTUFBTSxVQUFVLEdBQUcsTUFBTSxJQUFJLENBQUMsV0FBVyxDQUFBO2dCQUN6QyxNQUFNLE9BQU8sR0FBRyxNQUFNLFVBQVUsQ0FBQyxPQUFPLENBQUM7b0JBQ3ZDLEdBQUcsRUFBRSxJQUFJLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxDQUFDO29CQUMvQixHQUFHLEVBQUU7d0JBQ0gsRUFBRSxPQUFPLEVBQUUsRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFLEVBQUU7d0JBQy9CLEVBQUUsT0FBTyxFQUFFLEVBQUUsR0FBRyxFQUFFLElBQUksSUFBSSxFQUFFLEVBQUUsRUFBRTtxQkFDakM7aUJBQ0YsQ0FBQyxDQUFBO2dCQUNGLElBQUksSUFBSSxDQUFDLE1BQU0sSUFBSSxPQUFPLEVBQUU7b0JBQzFCLE1BQU0sSUFBSSxDQUFDLGNBQWMsQ0FDdEIsT0FBMEMsQ0FDNUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFBO2lCQUNoQztnQkFDRCxNQUFNLENBQUMsR0FDTCxPQUFPLElBQUksSUFBSSxDQUFDLGtCQUFrQixDQUFDLFdBQVcsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUE7Z0JBQ2pFLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxVQUFVLEdBQUcsQ0FBQyxLQUFJLE9BQU8sYUFBUCxPQUFPLHVCQUFQLE9BQU8sQ0FBRSxZQUFZLENBQUEsRUFBRTtvQkFDeEQsQ0FBQyxDQUFDLFlBQVksR0FBRyxPQUFPLENBQUMsWUFBWSxDQUFBO2lCQUN0QztnQkFDRCxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxHQUFHLENBQUMsQ0FBQTtnQkFDckIsUUFBUSxDQUFDLElBQUksRUFBRSxDQUFDLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFBO2FBQzNDO1lBQUMsT0FBTyxLQUFLLEVBQUU7Z0JBQ2QsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFBO2FBQ2hCO1FBQ0gsQ0FBQyxDQUFDLEVBQUUsQ0FBQTtJQUNOLENBQUM7SUFFRDs7OztPQUlHO0lBQ0gsR0FBRyxDQUNELEdBQVcsRUFDWCxPQUE0QixFQUM1QixXQUErQixJQUFJO1FBRW5DLENBQUM7UUFBQSxDQUFDLEtBQUssSUFBSSxFQUFFOztZQUNYLElBQUk7Z0JBQ0YsS0FBSyxDQUFDLGtCQUFrQixHQUFHLEVBQUUsQ0FBQyxDQUFBO2dCQUM5Qix1RUFBdUU7Z0JBQ3ZFLGFBQWE7Z0JBQ2IsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLFVBQVUsR0FBRyxDQUFDLEtBQUksT0FBTyxhQUFQLE9BQU8sdUJBQVAsT0FBTyxDQUFFLFlBQVksQ0FBQSxFQUFFO29CQUN4RCxhQUFhO29CQUNiLE9BQU8sT0FBTyxDQUFDLFlBQVksQ0FBQTtpQkFDNUI7Z0JBQ0QsTUFBTSxDQUFDLEdBQXdCO29CQUM3QixHQUFHLEVBQUUsSUFBSSxDQUFDLGdCQUFnQixDQUFDLEdBQUcsQ0FBQztvQkFDL0IsT0FBTyxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDO2lCQUNwRCxDQUFBO2dCQUNELGtCQUFrQjtnQkFDbEIsVUFBSSxPQUFPLGFBQVAsT0FBTyx1QkFBUCxPQUFPLENBQUUsTUFBTSwwQ0FBRSxPQUFPLEVBQUU7b0JBQzVCLENBQUMsQ0FBQyxPQUFPLEdBQUcsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQTtpQkFDN0M7cUJBQU07b0JBQ0wsaURBQWlEO29CQUNqRCx1REFBdUQ7b0JBQ3ZELDJCQUEyQjtvQkFDM0IsRUFBRTtvQkFDRixpREFBaUQ7b0JBQ2pELHlEQUF5RDtvQkFDekQsMkNBQTJDO29CQUMzQyxDQUFDLENBQUMsT0FBTyxHQUFHLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsR0FBRyxJQUFJLENBQUMsQ0FBQTtpQkFDM0Q7Z0JBQ0QsdUJBQXVCO2dCQUN2QixJQUFJLElBQUksQ0FBQyxPQUFPLENBQUMsVUFBVSxHQUFHLENBQUMsRUFBRTtvQkFDL0IsQ0FBQyxDQUFDLFlBQVksR0FBRyxJQUFJLElBQUksRUFBRSxDQUFBO2lCQUM1QjtnQkFDRCxJQUFJLElBQUksQ0FBQyxNQUFNLEVBQUU7b0JBQ2YsTUFBTSxTQUFTLEdBQUcsY0FBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUE7b0JBQ25FLE1BQU0sSUFBSSxHQUFHLE1BQU0sU0FBUyxDQUMxQixJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxNQUFnQixFQUNwQyxDQUFDLENBQUMsT0FBTyxDQUNWLENBQUMsS0FBSyxDQUFDLENBQUMsR0FBRyxFQUFFLEVBQUU7d0JBQ2QsTUFBTSxJQUFJLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQTtvQkFDdEIsQ0FBQyxDQUFDLENBQUE7b0JBQ0YsQ0FBQyxDQUFDLE9BQU8sR0FBSSxJQUF1QyxDQUFBO2lCQUNyRDtnQkFDRCxNQUFNLFVBQVUsR0FBRyxNQUFNLElBQUksQ0FBQyxXQUFXLENBQUE7Z0JBQ3pDLE1BQU0sT0FBTyxHQUFHLE1BQU0sVUFBVSxDQUFDLFNBQVMsQ0FDeEMsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUNkLEVBQUUsSUFBSSxFQUFFLENBQUMsRUFBRSxFQUNYO29CQUNFLE1BQU0sRUFBRSxJQUFJO29CQUNaLFlBQVksRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLHFCQUFxQjtpQkFDakQsQ0FDRixDQUFBO2dCQUNELElBQUksT0FBTyxDQUFDLGFBQWEsR0FBRyxDQUFDLEVBQUU7b0JBQzdCLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLEdBQUcsQ0FBQyxDQUFBO2lCQUN6QjtxQkFBTTtvQkFDTCxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxHQUFHLENBQUMsQ0FBQTtpQkFDekI7Z0JBQ0QsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsR0FBRyxDQUFDLENBQUE7YUFDdEI7WUFBQyxPQUFPLEtBQUssRUFBRTtnQkFDZCxPQUFPLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQTthQUN2QjtZQUNELE9BQU8sUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFBO1FBQ3ZCLENBQUMsQ0FBQyxFQUFFLENBQUE7SUFDTixDQUFDO0lBRUQsS0FBSyxDQUNILEdBQVcsRUFDWCxPQUFzRCxFQUN0RCxXQUErQixJQUFJO1FBRW5DLENBQUM7UUFBQSxDQUFDLEtBQUssSUFBSSxFQUFFOztZQUNYLElBQUk7Z0JBQ0YsS0FBSyxDQUFDLG9CQUFvQixHQUFHLEVBQUUsQ0FBQyxDQUFBO2dCQUNoQyxNQUFNLFlBQVksR0FJZCxFQUFFLENBQUE7Z0JBQ04sTUFBTSxVQUFVLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxVQUFVLEdBQUcsSUFBSSxDQUFBO2dCQUNqRCxNQUFNLFlBQVksR0FBRyxPQUFPLENBQUMsWUFBWTtvQkFDdkMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsT0FBTyxFQUFFO29CQUNoQyxDQUFDLENBQUMsQ0FBQyxDQUFBO2dCQUNMLE1BQU0sV0FBVyxHQUFHLElBQUksSUFBSSxFQUFFLENBQUE7Z0JBRTlCLCtEQUErRDtnQkFDL0QsNERBQTREO2dCQUM1RCxzREFBc0Q7Z0JBQ3RELElBQUksVUFBVSxHQUFHLENBQUMsSUFBSSxZQUFZLEdBQUcsQ0FBQyxFQUFFO29CQUN0QyxNQUFNLFdBQVcsR0FBRyxXQUFXLENBQUMsT0FBTyxFQUFFLEdBQUcsWUFBWSxDQUFBO29CQUN4RCxJQUFJLFdBQVcsR0FBRyxVQUFVLEVBQUU7d0JBQzVCLEtBQUssQ0FBQyx5QkFBeUIsR0FBRyxFQUFFLENBQUMsQ0FBQTt3QkFDckMsT0FBTyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUE7cUJBQ3RCO29CQUNELFlBQVksQ0FBQyxZQUFZLEdBQUcsV0FBVyxDQUFBO2lCQUN4QztnQkFFRCxVQUFJLE9BQU8sYUFBUCxPQUFPLHVCQUFQLE9BQU8sQ0FBRSxNQUFNLDBDQUFFLE9BQU8sRUFBRTtvQkFDNUIsWUFBWSxDQUFDLE9BQU8sR0FBRyxJQUFJLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFBO2lCQUN4RDtxQkFBTTtvQkFDTCxZQUFZLENBQUMsT0FBTyxHQUFHLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsR0FBRyxJQUFJLENBQUMsQ0FBQTtpQkFDdEU7Z0JBQ0QsTUFBTSxVQUFVLEdBQUcsTUFBTSxJQUFJLENBQUMsV0FBVyxDQUFBO2dCQUN6QyxNQUFNLE9BQU8sR0FBRyxNQUFNLFVBQVUsQ0FBQyxTQUFTLENBQ3hDLEVBQUUsR0FBRyxFQUFFLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLENBQUMsRUFBRSxFQUNuQyxFQUFFLElBQUksRUFBRSxZQUFZLEVBQUUsRUFDdEIsRUFBRSxZQUFZLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxxQkFBcUIsRUFBRSxDQUNyRCxDQUFBO2dCQUNELElBQUksT0FBTyxDQUFDLFlBQVksS0FBSyxDQUFDLEVBQUU7b0JBQzlCLE9BQU8sUUFBUSxDQUFDLElBQUksS0FBSyxDQUFDLHFDQUFxQyxDQUFDLENBQUMsQ0FBQTtpQkFDbEU7cUJBQU07b0JBQ0wsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsR0FBRyxFQUFFLE9BQU8sQ0FBQyxDQUFBO29CQUNoQyxPQUFPLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQTtpQkFDdEI7YUFDRjtZQUFDLE9BQU8sS0FBSyxFQUFFO2dCQUNkLE9BQU8sUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFBO2FBQ3ZCO1FBQ0gsQ0FBQyxDQUFDLEVBQUUsQ0FBQTtJQUNOLENBQUM7SUFFRDs7T0FFRztJQUNILEdBQUcsQ0FDRCxRQU1TO1FBRVQsQ0FBQztRQUFBLENBQUMsS0FBSyxJQUFJLEVBQUU7WUFDWCxJQUFJO2dCQUNGLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxDQUFBO2dCQUN6QixNQUFNLFVBQVUsR0FBRyxNQUFNLElBQUksQ0FBQyxXQUFXLENBQUE7Z0JBQ3pDLE1BQU0sUUFBUSxHQUFHLFVBQVUsQ0FBQyxJQUFJLENBQUM7b0JBQy9CLEdBQUcsRUFBRTt3QkFDSCxFQUFFLE9BQU8sRUFBRSxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUUsRUFBRTt3QkFDL0IsRUFBRSxPQUFPLEVBQUUsRUFBRSxHQUFHLEVBQUUsSUFBSSxJQUFJLEVBQUUsRUFBRSxFQUFFO3FCQUNqQztpQkFDRixDQUFDLENBQUE7Z0JBQ0YsTUFBTSxPQUFPLEdBQTBCLEVBQUUsQ0FBQTtnQkFDekMsSUFBSSxLQUFLLEVBQUUsTUFBTSxPQUFPLElBQUksUUFBUSxFQUFFO29CQUNwQyxJQUFJLElBQUksQ0FBQyxNQUFNLElBQUksT0FBTyxFQUFFO3dCQUMxQixNQUFNLElBQUksQ0FBQyxjQUFjLENBQ3RCLE9BQTBDLENBQzVDLENBQUE7cUJBQ0Y7b0JBQ0QsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFBO2lCQUNuRTtnQkFDRCxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQTtnQkFDekIsUUFBUSxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQTthQUN4QjtZQUFDLE9BQU8sS0FBSyxFQUFFO2dCQUNkLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQTthQUNoQjtRQUNILENBQUMsQ0FBQyxFQUFFLENBQUE7SUFDTixDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsT0FBTyxDQUFDLEdBQVcsRUFBRSxXQUErQixJQUFJO1FBQ3RELEtBQUssQ0FBQyxzQkFBc0IsR0FBRyxFQUFFLENBQUMsQ0FBQTtRQUNsQyxJQUFJLENBQUMsV0FBVzthQUNiLElBQUksQ0FBQyxDQUFDLFVBQVUsRUFBRSxFQUFFLENBQ25CLFVBQVUsQ0FBQyxTQUFTLENBQ2xCLEVBQUUsR0FBRyxFQUFFLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLENBQUMsRUFBRSxFQUNuQyxFQUFFLFlBQVksRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLHFCQUFxQixFQUFFLENBQ3JELENBQ0Y7YUFDQSxJQUFJLENBQUMsR0FBRyxFQUFFO1lBQ1QsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsR0FBRyxDQUFDLENBQUE7WUFDekIsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFBO1FBQ2hCLENBQUMsQ0FBQzthQUNELEtBQUssQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUE7SUFDbEMsQ0FBQztJQUVEOztPQUVHO0lBQ0gsTUFBTSxDQUFDLFFBQTRDO1FBQ2pELEtBQUssQ0FBQyxxQkFBcUIsQ0FBQyxDQUFBO1FBQzVCLElBQUksQ0FBQyxXQUFXO2FBQ2IsSUFBSSxDQUFDLENBQUMsVUFBVSxFQUFFLEVBQUUsQ0FBQyxVQUFVLENBQUMsY0FBYyxFQUFFLENBQUM7YUFDakQsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQy9CLGFBQWE7YUFDWixLQUFLLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFBO0lBQ2xDLENBQUM7SUFFRDs7T0FFRztJQUNILEtBQUssQ0FBQyxXQUErQixJQUFJO1FBQ3ZDLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQyxDQUFBO1FBQzNCLElBQUksQ0FBQyxXQUFXO2FBQ2IsSUFBSSxDQUFDLENBQUMsVUFBVSxFQUFFLEVBQUUsQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFLENBQUM7YUFDdkMsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQzthQUMxQixLQUFLLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFBO0lBQ2xDLENBQUM7SUFFRDs7T0FFRztJQUNILEtBQUs7UUFDSCxLQUFLLENBQUMsb0JBQW9CLENBQUMsQ0FBQTtRQUMzQixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxFQUFFLENBQUMsQ0FBQTtJQUM1QyxDQUFDO0NBQ0Y7QUFyYUQsNkJBcWFDIn0=